## General information

This example uses a small C program which uses a shared library specified by ourselves. It is compiled in a multi-stage Dockerfile. The Dockerfile uses a separate base image that compiles our library and executable using gcc.

The second stage of the Dockerfile retrieves the shared library and configures the dynamic bindings (especially for our custom shared library) in the image. The resulting library is located in `/usr/lib` (standard path for this).

The constant values for our paths (eventually generated by script) will have to include additionally the `/usr/lib`.

Our custom `step.py`, which uses the generated `main`, will try to call the main executable both with and without the loader. In the App Console of the pipeline, we can see that without the specific loader context, the shared library cannot be found.

### Build the c_image with its multi-stage Dockerfile

```sh
docker build -t c_image -f ./resources/Dockerfile .
```

### Create the image which includes building the file system for the image
```sh
CONTAINER_ID=$(docker create c_image) && echo $CONTAINER_ID
```

### Export the file system from the image and compress it as xz-file
```sh
docker export $CONTAINER_ID | xz > ./resources/distro_flat.xz
```

### Adapt the python example to the different environment (debian, C executable)
Use our script to generate the path into `contant.py`
```sh
docker run --rm -it -v "/$(pwd)/../../scripts:/scripts" c_image bash -c "cd /scripts && ./create_environment.sh" > src/constant.py
```
Additionally, for this example, we need the path to our shared libraries. Modify `OWN_LIBRARY_PATHS` to include it (see last line here).
```python
# Paths to all the necessary used library paths in your own filesystem must be adapted
OWN_LIBRARY_PATHS = "$(pwd)/usr/local/lib:$(pwd)/usr/local/lib/x86_64-linux-gnu:" \
                    "$(pwd)/lib/x86_64-linux-gnu:$(pwd)/usr/lib/x86_64-linux-gnu:" \
                    "$(pwd)/usr/lib"
```
The result (`src/contant.py`) should look like this:
```python
# Paths for the specific loader library from your own filesystem must be adapted
OWN_ELF_LOADER = "$(pwd)/lib/x86_64-linux-gnu/ld-2.31.so"
# Path to the required and installed fakechroot library must be adapted
OWN_PRELOAD = "$(pwd)/usr/lib/x86_64-linux-gnu/fakechroot/libfakechroot.so"
# Paths to all the necessary used library paths in your own filesystem must be adapted
OWN_LIBRARY_PATHS = "$(pwd)/usr/local/lib:$(pwd)/usr/local/lib/x86_64-linux-gnu:" \
                    "$(pwd)/lib/x86_64-linux-gnu:$(pwd)/usr/lib/x86_64-linux-gnu:" \
                    "$(pwd)/usr/lib"

# Path to the extracted filesystem relative to the current working directory after startup
OWN_FILE_SYSTEM_DIR = "./distro"
```

### Calling the executable from a python step
The generated executable can be called directly like a command line using the `run_with_own_loader` (see `src/step.py`).
For input data, we use the redirection of stdin `<<<`. Further aspects (certainly closer to reality) like passing specific parts of the payload or parsing data in the C program have not been addressed in this example.
```python
payload = str(document['payload'])
document['payload']['uppercase'] = run_with_own_loader('$(pwd)/opt/build/main <<< "' + payload + '"')
```

### Package the zip file for this example
Run following commands to prepare the content of the folder to zip.
```sh
mkdir -p ../../target
cp -r ../../src ../../executable-manifest.yaml ./resources ./src ../../target
```

If you have `zip` installed, you can run the following:
```
cd ../../target && zip -r C-example.zip * && cd -
```

Otherwise, you can also create you zip archive manually by selecting **the content** of the `target` folder (do not select the folder itself!) and adding it to a zip archive.

## Specific information

In this example, once you have built the image, you can run following command to test it
```sh
docker run --rm -it c_image bash -c './main <<< "anyStringInCamelCase"'
```

## Troubleshooting

### ldconfig does not work as expected

#### The libraries are not found even after a successful ldconfig
You can check if the library (in our case _upper_) has been correctly cached into the shared libraries list:
```shell
root@38c54a0132f7:/opt/build# ldconfig -p | grep upper
        libupper.so (libc6,x86-64) => /usr/lib/libupper.so
```
If not, assert that you have respected the naming conventions (`lib<yourLibraryName>.so`) and that it is located at a valid place (for example `/usr/lib`).

### The python commands using our executable give no output

#### The shared library could not be found
If you tested your script in the image successfully (see "Specific information" above), then the problem could be caused by a shared library which could not be found.

Be sure to specify in `constant.py` the folder where you put your library into and think about the dynamic working directory (`${pwd}`). In our case, it is `${pwd}/opt/utils`.

Also make sure to run your command **with own loader** if you use a shared library.

#### Problem with the _stdin_ input redirection
The redirection to stdin only supports characters. Any other conversion needs to be done in the executable code. In your `step.py`, make sure to avoid conflicts between single quotes, double quotes and backticks - as well in the python code as in the shell code.  
